//
// Copyright (c) 2010-2025 Antmicro
// Copyright (c) 2011-2015 Realtime Embedded
//
// This file is licensed under the MIT License.
// Full license text is available in 'licenses/MIT.txt'.
//
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;

using Antmicro.Migrant;
using Antmicro.Migrant.Hooks;
using Antmicro.Renode.Core;
using Antmicro.Renode.Debugging;
using Antmicro.Renode.Exceptions;
using Antmicro.Renode.Hooks;
using Antmicro.Renode.Logging;
using Antmicro.Renode.Logging.Profiling;
using Antmicro.Renode.Peripherals.Bus;
using Antmicro.Renode.Peripherals.CPU.Assembler;
using Antmicro.Renode.Peripherals.CPU.Disassembler;
using Antmicro.Renode.Peripherals.Miscellaneous;
using Antmicro.Renode.Utilities;
using Antmicro.Renode.Utilities.Binding;

using ELFSharp.ELF;

using Range = Antmicro.Renode.Core.Range;

namespace Antmicro.Renode.Peripherals.CPU
{
    public static class TranslationCPUHooksExtensions
    {
        public static void SetHookAtBlockBegin(this TranslationCPU cpu, [AutoParameter] IMachine m, string pythonScript)
        {
            var engine = new BlockPythonEngine(m, cpu, pythonScript);
            cpu.SetHookAtBlockBegin(engine.HookWithSize);
        }

        public static void SetHookAtBlockEnd(this TranslationCPU cpu, [AutoParameter] IMachine m, string pythonScript)
        {
            var engine = new BlockPythonEngine(m, cpu, pythonScript);
            cpu.SetHookAtBlockEnd(engine.HookWithSize);
        }
    }

    /// <summary>
    /// <see cref="TranslationCPU"/> implements <see cref="ICluster{T}"/> interface
    /// to seamlessly handle either cluster or CPU as a parameter to different methods.
    /// </summary>
    public abstract partial class TranslationCPU : BaseCPU, ICluster<TranslationCPU>, IGPIOReceiver, ICpuSupportingGdb, ICPUWithExternalMmu, ICPUWithMMU, INativeUnwindable, ICPUWithMetrics, ICPUWithMappedMemory, ICPUWithRegisters, ICPUWithMemoryAccessHooks, IControllableCPU, IHasPreservableState
    {
        public void AddHookAtInterruptBegin(Action<ulong> hook)
        {
            if(interruptBeginHook == null)
            {
                TlibSetInterruptBeginHookPresent(1u);
            }
            interruptBeginHook += hook;
        }

        public void MapMemory(IMappedSegment segment)
        {
            if(segment.StartingOffset > bitness.GetMaxAddress() || segment.StartingOffset + segment.Size - 1 > bitness.GetMaxAddress())
            {
                throw new RecoverableException("Could not map memory segment: it does not fit into address space");
            }

            using(machine?.ObtainPausedState(true))
            {
                currentMappings.Add(new SegmentMapping(segment));
                mappedMemory.Add(segment.GetRange());
                SetAccessMethod(segment.GetRange(), true);
            }
            this.NoisyLog("Registered memory at 0x{0:X}, size 0x{1:X}.", segment.StartingOffset, segment.Size);
        }

        public void RegisterAccessFlags(ulong startAddress, ulong size, bool isIoMemory = false)
        {
            TlibRegisterAccessFlagsForRange(startAddress, size, isIoMemory ? 1u : 0u);
        }

        public void SetMappedMemoryEnabled(Range range, bool enabled)
        {
            using(machine?.ObtainPausedState(true))
            {
                // Check if anything needs to be changed.
                if(enabled ? !disabledMemory.ContainsOverlappingRange(range) : disabledMemory.ContainsWholeRange(range))
                {
                    return;
                }

                if(enabled)
                {
                    disabledMemory.Remove(range);
                }
                else
                {
                    disabledMemory.Add(range);
                }
                SetAccessMethod(range, asMemory: enabled);
            }
        }

        public void UnmapMemory(Range range)
        {
            using(machine?.ObtainPausedState(true))
            {
                // when unmapping memory, two things have to be done
                // first is to flag address range as no-memory (that is, I/O)
                SetAccessMethod(range, asMemory: false);

                // remove mappings that are not used anymore
                currentMappings = currentMappings.
                    Where(x => TlibIsRangeMapped(x.Segment.StartingOffset, x.Segment.StartingOffset + x.Segment.Size) == 1).ToList();
                mappedMemory.Remove(range);
                RebuildMemoryMappings();
            }
        }

        public void SetPageAccessViaIo(ulong address)
        {
            TlibSetPageIoAccessed(address);
        }

        public void ClearPageAccessViaIo(ulong address)
        {
            TlibClearPageIoAccessed(address);
        }

        // this is just for easier usage in Monitor
        public void LogFunctionNames(bool value, bool removeDuplicates = false, bool useFunctionSymbolsOnly = true)
        {
            LogFunctionNames(value, string.Empty, removeDuplicates, useFunctionSymbolsOnly);
        }

        public ulong GetCurrentInstructionsCount()
        {
            return TlibGetTotalExecutedInstructions();
        }

        public void LogFunctionNames(bool value, string spaceSeparatedPrefixes = "", bool removeDuplicates = false, bool useFunctionSymbolsOnly = true)
        {
            if(!value)
            {
                logFunctionNamesCurrentState = null;
                SetInternalHookAtBlockBegin(null);
                return;
            }
            logFunctionNamesCurrentState = new LogFunctionNamesState(spaceSeparatedPrefixes, removeDuplicates, useFunctionSymbolsOnly);

            var prefixesAsArray = spaceSeparatedPrefixes.Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
            // using string builder here is due to performance reasons: test shows that string.Format is much slower
            var messageBuilder = new StringBuilder(256);

            Symbol previousSymbol = null;

            SetInternalHookAtBlockBegin((pc, size) =>
            {
                if(Bus.TryFindSymbolAt(pc, out var name, out var symbol, this, useFunctionSymbolsOnly))
                {
                    if(removeDuplicates && symbol == previousSymbol)
                    {
                        return;
                    }
                    previousSymbol = symbol;
                }

                if(spaceSeparatedPrefixes != "" && (name == null || !prefixesAsArray.Any(name.StartsWith)))
                {
                    return;
                }
                messageBuilder.Clear();
                this.Log(LogLevel.Info, messageBuilder.Append("Entering function ").Append(name ?? "without name").Append(" at 0x").Append(pc.ToString("X")).ToString());
            });
        }

        public RegisterValue GetRegisterUnsafe(int register)
        {
            // This is obsolete API, left here only for compatibility
            this.Log(LogLevel.Warning, "Using `GetRegisterUnsafe` API is obsolete. Please change to `GetRegister`.");
            return GetRegister(register);
        }

        public void NativeUnwind()
        {
            TlibUnwind();
        }

        public void ClearHookAtBlockBegin()
        {
            SetHookAtBlockBegin(null);
        }

        public void EnterSingleStepModeSafely(HaltArguments args)
        {
            // this method should only be called from CPU thread,
            // but we should check it anyway
            CheckCpuThreadId();

            ExecutionMode = ExecutionMode.SingleStep;

            UpdateHaltedState();
            InvokeHalted(args);
        }

        public void SetHookAtBlockBegin(Action<ulong, uint> hook)
        {
            using(machine?.ObtainPausedState(true))
            {
                if((hook == null) ^ (blockBeginUserHook == null))
                {
                    ClearTranslationCache();
                }
                blockBeginUserHook = hook;
                UpdateBlockBeginHookPresent();
            }
        }

        public void SetHookAtMemoryAccess(MemoryAccessHook hook)
        {
            TlibOnMemoryAccessEventEnabled(hook != null ? 1 : 0);
            memoryAccessHook = hook;
        }

        public void AddHookOnMmuFault(ExternalMmuFaultHook hook)
        {
            mmuFaultHook += hook;
        }

        public void RemoveHookOnMmuFault(ExternalMmuFaultHook hook)
        {
            mmuFaultHook -= hook;
        }

        public void AddHookAtInterruptEnd(Action<ulong> hook)
        {
            if(!Architecture.Contains("riscv") && !Architecture.Contains("arm") && !Architecture.Contains("sparc"))
            {
                throw new RecoverableException("Hooks at the end of interrupt are supported only in the RISC-V and ARM architectures");
            }

            if(interruptEndHook == null)
            {
                TlibSetInterruptEndHookPresent(1u);
            }
            interruptEndHook += hook;
        }

        public void LogCpuInterrupts(bool isEnabled)
        {
            if(isEnabled)
            {
                if(!isInterruptLoggingEnabled)
                {
                    AddHookAtInterruptBegin(LogCpuInterruptBegin);
                    AddHookAtInterruptEnd(LogCpuInterruptEnd);
                    isInterruptLoggingEnabled = true;
                }
            }
            else
            {
                RemoveHookAtInterruptBegin(LogCpuInterruptBegin);
                RemoveHookAtInterruptEnd(LogCpuInterruptEnd);
                isInterruptLoggingEnabled = false;
            }
        }

        public void AddHookAtWfiStateChange(Action<bool> hook)
        {
            if(wfiStateChangeHook == null)
            {
                TlibSetCpuWfiStateChangeHookPresent(1);
            }
            wfiStateChangeHook += hook;
        }

        public void RemoveHookAtWfiStateChange(Action<bool> hook)
        {
            wfiStateChangeHook -= hook;
            if(wfiStateChangeHook == null)
            {
                TlibSetCpuWfiStateChangeHookPresent(0);
            }
        }

        public void SetRegisterUnsafe(int register, RegisterValue value)
        {
            // This is obsolete API, left here only for compatibility
            this.Log(LogLevel.Warning, "Using `SetRegisterUnsafe` API is obsolete. Please change to `SetRegister`.");
            SetRegister(register, value);
        }

        public void AddHook(ulong addr, CpuAddressHook hook) => hooks.AddHook(addr, hook);

        public void RemoveHook(ulong addr, CpuAddressHook hook) => hooks.RemoveHook(addr, hook);

        public void OrderTranslationBlocksInvalidation(IntPtr start, IntPtr end, bool delayInvalidation = false)
        {
            if(disposing)
            {
                return;
            }

            lock(addressesToInvalidate)
            {
                // Address ranges are passed to tlib as interleaved pairs of start and end addresses
                addressesToInvalidate.Add(start);
                addressesToInvalidate.Add(end);
            }

            if(!delayInvalidation)
            {
                InvalidateTranslationBlocks();
            }
        }

        public void InvalidateTranslationBlocks()
        {
            lock(addressesToInvalidate)
            {
                var count = (ulong)addressesToInvalidate.Count / 2;
                if(count > 0)
                {
                    unsafe
                    {
                        fixed(void* addresses = addressesToInvalidate.ToArray())
                        {
                            TlibInvalidateTranslationBlocks((IntPtr)addresses, count);
                        }
                    }
                    addressesToInvalidate.Clear();
                }
            }
        }

        public void LoadPreservedState(object state)
        {
            if(!(state is TranslationCPUState cpuState))
            {
                throw new RecoverableException("Unexpected state received while loading preserved state");
            }
            isInterruptLoggingEnabled = cpuState.IsInterruptLoggingEnabled;
            if(cpuState.LogFunctionNamesState.HasValue)
            {
                var logFunctionNamesState = cpuState.LogFunctionNamesState.Value;
                LogFunctionNames(true, logFunctionNamesState.SpaceSeparatedPrefixes, logFunctionNamesState.RemoveDuplicates, logFunctionNamesState.UseFunctionSymbolsOnly);
            }
        }

        public object ExtractPreservedState()
        {
            return new TranslationCPUState(logFunctionNamesCurrentState, isInterruptLoggingEnabled);
        }

        public override void Dispose()
        {
            // Prevent trying to release unmanaged resources if we have already disposed them
            if(disposed)
            {
                return;
            }
            disposed = true;
            base.Dispose();
            profiler?.Dispose();
            localAtomicState?.Dispose();
        }

        public void SetHookAtBlockEnd(Action<ulong, uint> hook)
        {
            using(machine?.ObtainPausedState(true))
            {
                if((hook == null) ^ (blockFinishedHook == null))
                {
                    ClearTranslationCache();
                    TlibSetBlockFinishedHookPresent(hook != null ? 1u : 0u);
                }
                blockFinishedHook = hook;
            }
        }

        public virtual void OnGPIO(int number, bool value)
        {
            lock(lck)
            {
                if(ThreadSentinelEnabled)
                {
                    CheckIfOnSynchronizedThread();
                }
                this.NoisyLog("IRQ {0}, value {1}", number, value);
                // as we are waiting for an interrupt we should, obviously, not mask it
                if(started && (lastTlibResult == TlibExecutionResult.WaitingForInterrupt || !(DisableInterruptsWhileStepping && IsSingleStepMode)))
                {
                    TlibSetIrqWrapped(number, value);
                    if(EmulationManager.Instance.CurrentEmulation.Mode != Emulation.EmulationMode.SynchronizedIO)
                    {
                        sleeper.Interrupt();
                    }
                }
            }
        }

        public void EnableReadCache(ulong accessAddress, ulong lowerAccessCount, ulong upperAccessCount = 0)
        {
            if(lowerAccessCount == 0)
            {
                throw new RecoverableException("Lower access count to address cannot be zero!");
            }
            if((upperAccessCount != 0) && ((upperAccessCount <= lowerAccessCount)))
            {
                throw new RecoverableException("Upper access count to address has to be bigger than lower access count!");
            }
            TlibEnableReadCache(accessAddress, lowerAccessCount, upperAccessCount);
        }

        public bool RequestTranslationBlockRestart(bool quiet = false)
        {
            if(!OnPossessedThread)
            {
                if(!quiet)
                {
                    this.Log(LogLevel.Error, "Translation block restart should be requested from CPU thread only. Ignoring the operation.");
                }
                return false;
            }
            return pauseGuard.RequestTranslationBlockRestart(quiet);
        }

        public uint AssembleBlock(ulong addr, string instructions, uint flags = 0)
        {
            if(Assembler == null)
            {
                throw new RecoverableException("Assembler not available");
            }

            // Instruction fetch access used as we want to be able to write even pages mapped for execution only
            // We don't care if translation fails here (the address is unchanged in this case)
            TryTranslateAddress(addr, MpuAccess.InstructionFetch, out addr);

            var result = Assembler.AssembleBlock(addr, instructions, flags);
            Bus.WriteBytes(result, addr, true, context: this);
            return (uint)result.Length;
        }

        public uint GetMmuWindowPrivileges(ulong id)
        {
            return AssertMmuEnabled() ? TlibGetWindowPrivileges(id) : 0;
        }

        public ulong GetMmuWindowEnd(ulong id)
        {
            return AssertMmuEnabled() ? TlibGetMmuWindowEnd(id) : 0;
        }

        public ulong GetMmuWindowStart(ulong id)
        {
            return AssertMmuEnabled() ? TlibGetMmuWindowStart(id) : 0;
        }

        public ulong GetMmuWindowAddend(ulong id)
        {
            return AssertMmuEnabled() ? TlibGetMmuWindowAddend(id) : 0;
        }

        public void SetMmuWindowPrivileges(ulong id, ExternalMmuBase.Privilege permissions)
        {
            if(AssertMmuEnabled())
            {
                TlibSetWindowPrivileges(id, (uint)permissions);
            }
        }

        public void SetMmuWindowEnd(ulong id, ulong endAddress)
        {
            if(AssertMmuEnabled() && AssertMmuWindowAddressInRange(endAddress, inclusiveRange: true))
            {
                bool useInclusiveEndRange = false;
                // Overflow on 64bits currently not possible due to type constraints
                if(this.bitness == CpuBitness.Bits32)
                {
                    useInclusiveEndRange = ((endAddress - 1) == UInt32.MaxValue);
                }

                if(useInclusiveEndRange)
                {
                    endAddress -= 1;
                }

                this.DebugLog("Setting range end to {0} addr 0x{1:x}", useInclusiveEndRange ? "inclusive" : "exclusive", endAddress);
                TlibSetMmuWindowEnd(id, endAddress, useInclusiveEndRange ? 1u : 0u);
            }
        }

        public void SetMmuWindowStart(ulong id, ulong startAddress)
        {
            if(AssertMmuEnabled() && AssertMmuWindowAddressInRange(startAddress))
            {
                TlibSetMmuWindowStart(id, startAddress);
            }
        }

        public void SetMmuWindowAddend(ulong id, ulong addend)
        {
            if(AssertMmuEnabled())
            {
                TlibSetMmuWindowAddend(id, addend);
            }
        }

        public void ResetMmuWindow(ulong id)
        {
            if(AssertMmuEnabled())
            {
                TlibResetMmuWindow(id);
            }
        }

        public void ResetMmuWindowsCoveringAddress(ulong address)
        {
            if(AssertMmuEnabled())
            {
                TlibResetMmuWindowsCoveringAddress(address);
            }
        }

        public void ResetAllMmuWindows()
        {
            if(AssertMmuEnabled())
            {
                TlibResetAllMmuWindows();
            }
        }

        public ulong AcquireExternalMmuWindow(ExternalMmuBase.Privilege type)
        {
            if(AssertMmuEnabled())
            {
                return TlibAcquireMmuWindow((uint)type);
            }
            return 0; // unreachable (assert throws in this case)
        }

        public void EnableExternalWindowMmu(bool value)
        {
            EnableExternalWindowMmu(value ? ExternalMmuPosition.Replace : ExternalMmuPosition.None);
        }

        public void EnableExternalWindowMmu(ExternalMmuPosition position)
        {
            TlibEnableExternalWindowMmu((uint)position);
            externalMmuPosition = position;
        }

        public void FlushTlb()
        {
            TlibFlushTlb(OnPossessedThread);
        }

        public void RaiseException(uint exceptionId)
        {
            TlibRaiseException(exceptionId);
        }

        public override string ToString()
        {
            return $"[CPU: {this.GetCPUThreadName(machine)}]";
        }

        public void FlushTlbPage(UInt64 address)
        {
            TlibFlushPage(address);
        }

        public override void Reset()
        {
            base.Reset();
            isInterruptLoggingEnabled = false;
            TlibReset();
            ResetOpcodesCounters();
            profiler?.Dispose();
        }

        /// <summary>
        /// Attempts to translate a logical (virtual) address to a physical address for the specified access type.
        /// </summary>
        /// <param name="logicalAddress">The logical (virtual) address to be translated.</param>
        /// <param name="accessType">The type of access (read, write, fetch), represented as an <see cref="MpuAccess"/> value.</param>
        /// <param name="physicalAddress">At return, contains the translated physical address if the translation is successful.
        /// If there is no page table entry for the requested logical address, this output will contain the original logical address.
        /// </param>
        /// <returns>
        /// <c>true</c> if the translation was successful; otherwise, <c>false</c>. In this case the result is the original address.
        /// </returns>
        public bool TryTranslateAddress(ulong logicalAddress, MpuAccess accessType, out ulong physicalAddress)
        {
            try
            {
                physicalAddress = TranslateAddress(logicalAddress, accessType);
                return true;
            }
            catch(RecoverableException)
            {
                physicalAddress = logicalAddress;
                return false;
            }
        }

        /// <summary>
        /// Translates a logical (virtual) address to a physical address for the specified access type.
        /// </summary>
        /// <param name="logicalAddress">The logical (virtual) address to be translated.</param>
        /// <param name="accessType">The type of access (read, write, fetch), represented as an <see cref="MpuAccess"/> value.</param>
        /// <returns>
        /// The translated physical address if the translation was successful; otherwise, <c>ulong.MaxValue</c>.
        /// </returns>
        public ulong TranslateAddress(ulong logicalAddress, MpuAccess accessType)
        {
            var physicalAddress = TlibTranslateToPhysicalAddress(logicalAddress, (uint)accessType);
            if(physicalAddress == ulong.MaxValue)
            {
                throw new RecoverableException($"Failed to translate address: 0x{logicalAddress:X}");
            }
            return physicalAddress;
        }

        public void EnableProfiling()
        {
            AddHookAtInterruptBegin(exceptionIndex =>
            {
                machine.Profiler.Log(new ExceptionEntry(exceptionIndex));
            });

            SetHookAtMemoryAccess((_, operation, __, physicalAddress, ___, value) =>
            {
                switch(operation)
                {
                case MemoryOperation.MemoryIORead:
                case MemoryOperation.MemoryIOWrite:
                    machine.Profiler?.Log(new PeripheralEntry((byte)operation, physicalAddress));
                    break;
                case MemoryOperation.MemoryRead:
                case MemoryOperation.MemoryWrite:
                    machine.Profiler?.Log(new MemoryEntry((byte)operation));
                    break;
                }
            });
        }

        public void RemoveAllHooks() => hooks.RemoveAllHooks();

        public void RequestReturn()
        {
            TlibSetReturnRequest();
        }

        public override void SyncTime()
        {
            if(!OnPossessedThread)
            {
                this.Log(LogLevel.Error, "Syncing time should be done from CPU thread only. Ignoring the operation");
                return;
            }

            var numberOfExecutedInstructions = TlibGetExecutedInstructions();
            this.Trace($"CPU executed {numberOfExecutedInstructions} instructions and time synced");
            ReportProgress(numberOfExecutedInstructions);
        }

        public void ActivateNewHooks() => hooks.ActivateNewHooks();

        public string DisassembleBlock(ulong addr = ulong.MaxValue, uint blockSize = 40, uint flags = 0)
        {
            if(Disassembler == null)
            {
                throw new RecoverableException("Disassembly engine not available");
            }
            if(addr == ulong.MaxValue)
            {
                addr = PC;
            }

            // Instruction fetch access used as we want to be able to read even pages mapped for execution only
            // We don't care if translation fails here (the address is unchanged in this case)
            TryTranslateAddress(addr, MpuAccess.InstructionFetch, out addr);

            var opcodes = Bus.ReadBytes(addr, (int)blockSize, true, context: this);
            Disassembler.DisassembleBlock(addr, opcodes, flags, out var result);
            return result;
        }

        public override ExecutionResult ExecuteInstructions(ulong numberOfInstructionsToExecute, out ulong numberOfExecutedInstructions)
        {
            ActivateNewHooks();

            try
            {
                while(actionsToExecuteOnCpuThread.TryDequeue(out var queuedAction))
                {
                    queuedAction();
                }

                pauseGuard.Enter();
                lastTlibResult = (TlibExecutionResult)TlibExecute(checked((int)numberOfInstructionsToExecute));
                pauseGuard.Leave();
            }
            catch(CpuAbortException)
            {
                this.NoisyLog("CPU abort detected, halting.");
                InvokeHalted(new HaltArguments(HaltReason.Abort, this));
                return ExecutionResult.Aborted;
            }
            finally
            {
                numberOfExecutedInstructions = TlibGetExecutedInstructions();
                if(numberOfExecutedInstructions == 0)
                {
                    this.Trace($"Asked tlib to execute {numberOfInstructionsToExecute}, but did nothing");
                }
                DebugHelper.Assert(numberOfExecutedInstructions <= numberOfInstructionsToExecute, "tlib executed more instructions than it was asked to");
            }

            switch(lastTlibResult)
            {
            case TlibExecutionResult.Ok:
                return ExecutionResult.Ok;

            case TlibExecutionResult.WaitingForInterrupt:
                return ExecutionResult.WaitingForInterrupt;

            case TlibExecutionResult.ExternalMmuFault:
                return ExecutionResult.ExternalMmuFault;

            case TlibExecutionResult.StoppedAtBreakpoint:
                return ExecutionResult.StoppedAtBreakpoint;

            case TlibExecutionResult.StoppedAtWatchpoint:
                return ExecutionResult.StoppedAtWatchpoint;

            case TlibExecutionResult.ReturnRequested:
                return ExecutionResult.Interrupted;

            default:
                throw new Exception();
            }
        }

        public void SetBroadcastDirty(bool enable)
        {
            TlibSetBroadcastDirty(enable ? 1 : 0);
        }

        public void ClearTranslationCache()
        {
            using(machine?.ObtainPausedState(true))
            {
                TlibInvalidateTranslationCache();
            }
        }

        public void RemoveHooksAt(ulong addr) => hooks.RemoveHooksAt(addr);

        public void RemoveHooks(CpuAddressHook hook) => hooks.RemoveHooks(hook);

        public new IEnumerator<TranslationCPU> GetEnumerator()
        {
            return Clustered.GetEnumerator();
        }

        // This method is overriden only to add [Export] attribute
        [Export]
        public override uint CheckExternalPermissions(ulong address)
        {
            return 1;
        }

        public abstract void SetRegister(int register, RegisterValue value);

        public abstract RegisterValue GetRegister(int register);

        public abstract IEnumerable<CPURegister> GetRegisters();

        public string PreservableName => $"TranslationCPU:{this.GetName()}";

        public LLVMDisassembler Disassembler => disassembler;

        public uint PageSize
        {
            get
            {
                return TlibGetPageSize();
            }
        }

        public bool TbCacheEnabled
        {
            get
            {
                return TlibGetTbCacheEnabled() != 0;
            }

            set
            {
                TlibSetTbCacheEnabled(value ? 1u : 0u);
            }
        }

        public bool SyncPCEveryInstructionDisabled
        {
            get
            {
                return TlibGetSyncPcEveryInstructionDisabled() != 0;
            }

            set
            {
                TlibSetSyncPcEveryInstructionDisabled(value ? 1u : 0u);
            }
        }

        public bool ChainingEnabled
        {
            get
            {
                return TlibGetChainingEnabled() != 0;
            }

            set
            {
                TlibSetChainingEnabled(value ? 1u : 0u);
            }
        }

        public int MaximumBlockSize
        {
            get
            {
                return checked((int)TlibGetMaximumBlockSize());
            }

            set
            {
                TlibSetMaximumBlockSize(checked((uint)value));
                ClearTranslationCache();
            }
        }

        /// <summary>
        /// The value is used to convert instructions count to cycles, e.g.:
        /// * for RISC-V CYCLE and MCYCLE CSRs
        /// * in ARM Performance Monitoring Unit
        /// </summary>
        public decimal CyclesPerInstruction
        {
            get
            {
                return checked(TlibGetMillicyclesPerInstruction() / 1000m);
            }

            set
            {
                if(value <= 0)
                {
                    throw new RecoverableException("Value must be a positive number.");
                }
                var millicycles = value * 1000m;
                if(millicycles % 1m != 0)
                {
                    throw new RecoverableException("Value's precision can't be greater than 0.001");
                }
                TlibSetMillicyclesPerInstruction(checked((uint)millicycles));
            }
        }

        public bool LogTranslationBlockFetch
        {
            set
            {
                if(value)
                {
                    RenodeAttachLogTranslationBlockFetch(Marshal.GetFunctionPointerForDelegate(onTranslationBlockFetch));
                }
                else
                {
                    RenodeAttachLogTranslationBlockFetch(IntPtr.Zero);
                }
                logTranslationBlockFetchEnabled = value;
            }

            get
            {
                return logTranslationBlockFetchEnabled;
            }
        }

        // This value should only be read in CPU hooks (during execution of translated code).
        public uint CurrentBlockDisassemblyFlags => TlibGetCurrentTbDisasFlags();

        public uint ExternalMmuWindowsCount => TlibGetMmuWindowsCount();

        public bool ThreadSentinelEnabled { get; set; }

        public override ulong ExecutedInstructions { get { return TlibGetTotalExecutedInstructions(); } }

        public int Slot { get { if(!slot.HasValue) slot = machine.SystemBus.GetCPUSlot(this); return slot.Value; } private set { slot = value; } }

        public string LogFile
        {
            get { return logFile; }

            set
            {
                logFile = value;
                LogTranslatedBlocks = (value != null);

                if(value == null)
                {
                    return;
                }

                try
                {
                    // truncate the file
                    File.WriteAllText(logFile, string.Empty);
                }
                catch(Exception e)
                {
                    throw new RecoverableException($"There was a problem when preparing the log file {logFile}: {e.Message}");
                }
            }
        }

        public override ExecutionMode ExecutionMode
        {
            get
            {
                return base.ExecutionMode;
            }

            set
            {
                base.ExecutionMode = value;
                UpdateBlockBeginHookPresent();
            }
        }

        public override RegisterValue PC
        {
            get
            {
                throw new NotImplementedException();
            }

            set
            {
                throw new NotImplementedException();
            }
        }

        public bool LogTranslatedBlocks
        {
            get
            {
                return logTranslatedBlocks;
            }

            set
            {
                if(LogFile == null && value)
                {
                    throw new RecoverableException("Log file not set. Nothing will be logged.");
                }
                logTranslatedBlocks = value;
                TlibSetOnBlockTranslationEnabled(value ? 1 : 0);
            }
        }

        public uint IRQ { get { return TlibIsIrqSet(); } }

        public LLVMAssembler Assembler => assembler;

        // TODO: improve this when backend/analyser stuff is done

        public bool UpdateContextOnLoadAndStore { get; set; }

        public override ulong SkipInstructions
        {
            get => base.SkipInstructions;
            protected set
            {
                if(!OnPossessedThread)
                {
                    this.Log(LogLevel.Error, "Changing SkipInstructions should be only done on CPU thread, ignoring");
                    return;
                }

                base.SkipInstructions = value;
                // This will be imprecise when we change SkipInstructions before end of the translation block
                // as TlibSetReturnRequest doesn't finish current translation block
                TlibSetReturnRequest();
            }
        }

        public bool DisableInterruptsWhileStepping { get; set; }

        public abstract List<GDBFeatureDescriptor> GDBFeatures { get; }

        public abstract string GDBArchitecture { get; }

        protected TranslationCPU(string cpuType, IMachine machine, Endianess endianness, CpuBitness bitness = CpuBitness.Bits32)
        : this(0, cpuType, machine, endianness, bitness)
        {
        }

        protected TranslationCPU(uint id, string cpuType, IMachine machine, Endianess endianness, CpuBitness bitness = CpuBitness.Bits32, bool useMachineAtomicState = true)
            : base(id, cpuType, machine, endianness, bitness)
        {
            if(!useMachineAtomicState)
            {
                // Create unique instance of atomic state for cpu that should not use the shared state.
                localAtomicState = new AtomicState(this, AtomicState.MinimalStoreTableBits);
            }

            atomicId = -1;
            pauseGuard = new CpuThreadPauseGuard(this);
            decodedIrqs = new Dictionary<Interrupt, HashSet<int>>();
            hooks = new HookDescriptor(this);
            currentMappings = new List<SegmentMapping>();
            this.useMachineAtomicState = useMachineAtomicState;
            InitializeRegisters();
            Init(afterDeserialization: false);
            InitDisas();
            Clustered = new TranslationCPU[] { this };
        }

        public new IEnumerable<ICluster<TranslationCPU>> Clusters { get; } = new List<ICluster<TranslationCPU>>(0);

        public new IEnumerable<TranslationCPU> Clustered { get; }

        [Export]
        protected void Free(IntPtr pointer)
        {
            memoryManager.Free(pointer);
        }

        [Export]
        protected virtual void LogAsCpu(int level, string s)
        {
            this.Log((LogLevel)level, s);
        }

        protected override void DisposeInner(bool silent = false)
        {
            base.DisposeInner(silent);
            TimeHandle?.Dispose();
            RemoveAllHooks();
            TlibDispose();
            RenodeFreeHostBlocks();
            binder.Dispose();
            if(!EmulationManager.DisableEmulationFilesCleanup)
            {
                File.Delete(libraryFile);
            }
            if(dirtyAddressesPtr != IntPtr.Zero)
            {
                memoryManager.Free(dirtyAddressesPtr);
            }
            memoryManager.CheckIfAllIsFreed();
        }

        protected virtual void InitializeRegisters()
        {
        }

        protected ulong GetCPUStateForMemoryTransaction()
        {
            return TlibGetCpuStateForMemoryTransaction();
        }

        protected virtual void AfterLoad(IntPtr statePtr)
        {
            TlibAfterLoad(statePtr);
        }

        protected virtual void BeforeSave(IntPtr statePtr)
        {
            TlibBeforeSave(statePtr);
        }

        [PostDeserialization]
        protected void InitDisas()
        {
            try
            {
                disassembler = new LLVMDisassembler(this);
            }
            catch(ArgumentOutOfRangeException)
            {
                this.Log(LogLevel.Warning, "Could not initialize disassembly engine");
            }
            try
            {
                assembler = new LLVMAssembler(this);
            }
            catch(ArgumentOutOfRangeException)
            {
                this.Log(LogLevel.Warning, "Could not initialize assembly engine");
            }
            dirtyAddressesPtr = IntPtr.Zero;
            addressesToInvalidate = new List<IntPtr>();
        }

        protected override bool UpdateHaltedState(bool ignoreExecutionMode = false)
        {
            if(!base.UpdateHaltedState(ignoreExecutionMode))
            {
                return false;
            }

            if(currentHaltedState)
            {
                TlibSetReturnRequest();
            }

            return true;
        }

        protected void WriteWordToBus(ulong offset, ulong value)
        {
            WriteWordToBus(offset, value, GetCPUStateForMemoryTransaction());
        }

        protected ulong ReadDoubleWordFromBus(ulong offset)
        {
            return ReadDoubleWordFromBus(offset, GetCPUStateForMemoryTransaction());
        }

        protected override void OnLeavingResetState()
        {
            base.OnLeavingResetState();
            TlibOnLeavingResetState();
        }

        protected override void RequestPause()
        {
            base.RequestPause();
            TlibSetReturnRequest();
        }

        protected override void InnerPause(bool onCpuThread, bool checkPauseGuard)
        {
            base.InnerPause(onCpuThread, checkPauseGuard);

            if(onCpuThread && checkPauseGuard)
            {
                pauseGuard.OrderPause();
            }
        }

        [Export]
        protected ulong ReadByteFromBus(ulong offset, ulong cpuState)
        {
            if(UpdateContextOnLoadAndStore)
            {
                TlibRestoreContext();
            }
            using(var guard = ObtainPauseGuardForReading(offset, SysbusAccessWidth.Byte))
            {
                // If the transaction was interrupted while handling a watchpoint, return 0 immediately to avoid
                // duplicating the access' side effect.
                return guard.InterruptTransaction
                    ? 0
                    : (ulong)machine.SystemBus.ReadByte(offset, this, cpuState);
            }
        }

        [Export]
        protected ulong ReadWordFromBus(ulong offset, ulong cpuState)
        {
            if(UpdateContextOnLoadAndStore)
            {
                TlibRestoreContext();
            }
            using(var guard = ObtainPauseGuardForReading(offset, SysbusAccessWidth.Word))
            {
                // If the transaction was interrupted while handling a watchpoint, return 0 immediately to avoid
                // duplicating the access' side effect.
                return guard.InterruptTransaction
                    ? 0
                    : (ulong)machine.SystemBus.ReadWord(offset, this, cpuState);
            }
        }

        [Export]
        protected ulong ReadDoubleWordFromBus(ulong offset, ulong cpuState)
        {
            if(UpdateContextOnLoadAndStore)
            {
                TlibRestoreContext();
            }
            using(var guard = ObtainPauseGuardForReading(offset, SysbusAccessWidth.DoubleWord))
            {
                // If the transaction was interrupted while handling a watchpoint, return 0 immediately to avoid
                // duplicating the access' side effect.
                return guard.InterruptTransaction
                    ? 0
                    : machine.SystemBus.ReadDoubleWord(offset, this, cpuState);
            }
        }

        [Export]
        protected ulong ReadQuadWordFromBus(ulong offset, ulong cpuState)
        {
            if(UpdateContextOnLoadAndStore)
            {
                TlibRestoreContext();
            }
            using(var guard = ObtainPauseGuardForReading(offset, SysbusAccessWidth.QuadWord))
            {
                // If the transaction was interrupted while handling a watchpoint, return 0 immediately to avoid
                // duplicating the access' side effect.
                return guard.InterruptTransaction
                    ? 0
                    : machine.SystemBus.ReadQuadWord(offset, this, cpuState);
            }
        }

        [Export]
        protected void WriteByteToBus(ulong offset, ulong value, ulong cpuState)
        {
            if(UpdateContextOnLoadAndStore)
            {
                TlibRestoreContext();
            }
            using(var guard = ObtainPauseGuardForWriting(offset, SysbusAccessWidth.Byte, value))
            {
                // If the transaction was interrupted while handling a watchpoint, don't perform the write to avoid
                // duplicating the access' side effect.
                if(!guard.InterruptTransaction)
                {
                    machine.SystemBus.WriteByte(offset, unchecked((byte)value), this, cpuState);
                }
            }
        }

        [Export]
        protected void WriteWordToBus(ulong offset, ulong value, ulong cpuState)
        {
            if(UpdateContextOnLoadAndStore)
            {
                TlibRestoreContext();
            }
            using(var guard = ObtainPauseGuardForWriting(offset, SysbusAccessWidth.Word, value))
            {
                // If the transaction was interrupted while handling a watchpoint, don't perform the write to avoid
                // duplicating the access' side effect.
                if(!guard.InterruptTransaction)
                {
                    machine.SystemBus.WriteWord(offset, unchecked((ushort)value), this, cpuState);
                }
            }
        }

        [Export]
        protected void WriteDoubleWordToBus(ulong offset, ulong value, ulong cpuState)
        {
            if(UpdateContextOnLoadAndStore)
            {
                TlibRestoreContext();
            }
            using(var guard = ObtainPauseGuardForWriting(offset, SysbusAccessWidth.DoubleWord, value))
            {
                // If the transaction was interrupted while handling a watchpoint, don't perform the write to avoid
                // duplicating the access' side effect.
                if(!guard.InterruptTransaction)
                {
                    machine.SystemBus.WriteDoubleWord(offset, (uint)value, this, cpuState);
                }
            }
        }

        [Export]
        protected void WriteQuadWordToBus(ulong offset, ulong value, ulong cpuState)
        {
            if(UpdateContextOnLoadAndStore)
            {
                TlibRestoreContext();
            }
            using(var guard = ObtainPauseGuardForWriting(offset, SysbusAccessWidth.QuadWord, value))
            {
                // If the transaction was interrupted while handling a watchpoint, don't perform the write to avoid
                // duplicating the access' side effect.
                if(!guard.InterruptTransaction)
                {
                    machine.SystemBus.WriteQuadWord(offset, value, this, cpuState);
                }
            }
        }

        protected ulong ReadByteFromBus(ulong offset)
        {
            return ReadByteFromBus(offset, GetCPUStateForMemoryTransaction());
        }

        protected ulong ReadWordFromBus(ulong offset)
        {
            return ReadWordFromBus(offset, GetCPUStateForMemoryTransaction());
        }

        protected ulong ReadQuadWordFromBus(ulong offset)
        {
            return ReadQuadWordFromBus(offset, GetCPUStateForMemoryTransaction());
        }

        protected void WriteByteToBus(ulong offset, ulong value)
        {
            WriteByteToBus(offset, value, GetCPUStateForMemoryTransaction());
        }

        /// <remarks>
        /// Be careful when using this method - the PauseGuard is used to verify if the precise pause is possible in a given context
        /// and as such, we should only obtain the guard when we for certain know it is.
        /// For example, precise pause is always possible if called from CPU loop in tlib
        /// </remarks>
        protected CpuThreadPauseGuard ObtainGenericPauseGuard()
        {
            pauseGuard.Initialize();
            return pauseGuard;
        }

        protected void WriteDoubleWordToBus(ulong offset, ulong value)
        {
            WriteDoubleWordToBus(offset, value, GetCPUStateForMemoryTransaction());
        }

        protected void WriteQuadWordToBus(ulong offset, ulong value)
        {
            WriteQuadWordToBus(offset, value, GetCPUStateForMemoryTransaction());
        }

        protected virtual string GetExceptionDescription(ulong exceptionIndex)
        {
            return $"Undecoded {exceptionIndex}";
        }

        protected override bool ExecutionFinished(ExecutionResult result)
        {
            if(result == ExecutionResult.StoppedAtBreakpoint)
            {
                this.Trace();
                hooks.Execute(PC);
                // it is necessary to deactivate hooks installed on this PC before
                // calling `tlib_execute` again to avoid a loop;
                // we need to do this because creating a breakpoint has caused special
                // exception-rising, block-breaking `trap` instruction to be
                // generated by the tcg;
                // in order to execute code after the breakpoint we must first remove
                // this `trap` and retranslate the code right after it;
                // this is achieved by deactivating the breakpoint (i.e., unregistering
                // from tlib, but keeping it in C#), executing the beginning of the next
                // block and registering the breakpoint again in the OnBlockBegin hook
                DeactivateHooks(PC);
                return true;
            }
            else if(result == ExecutionResult.StoppedAtWatchpoint)
            {
                this.Trace();
                // If we stopped at a watchpoint we must've been in the process
                // of executing an instruction which accesses memory.
                // That means that if there have been any hooks added for the current PC,
                // they were already executed, and the PC has been moved back by one instruction.
                // We don't want to execute them again, so we disable them temporarily.
                DeactivateHooks(PC);
                return true;
            }
            else if(result == ExecutionResult.WaitingForInterrupt)
            {
                if(InDebugMode || neverWaitForInterrupt)
                {
                    // NIP always points to the next instruction, on all emulated cores. If this behavior changes, this needs to change as well.
                    this.Trace("Clearing WaitForInterrupt processor state.");
                    TlibCleanWfiProcState(); // Clean WFI state in the emulated core
                    return true;
                }
            }

            return false;
        }

        protected abstract Interrupt DecodeInterrupt(int number);

        protected virtual bool IsSecondary
        {
            get
            {
                return Slot > 0;
            }
        }

        // 649:  Field '...' is never assigned to, and will always have its default value null
#pragma warning disable 649
        [Import]
        protected Func<ulong> TlibGetExecutedInstructions;

        [Import]
        protected Func<ulong, uint, ulong> TlibTranslateToPhysicalAddress;

        [Import]
        protected Action TlibSetReturnRequest;

        [Import]
        protected Action<int> TlibRequestTranslationBlockInterrupt;

        [Import]
        protected readonly Action<ulong, ulong, uint> TlibEnableExternalPermissionHandlerForRange;
#pragma warning restore 649

        /*
            Increments each time a new translation library resource is created.
            This counter marks each new instance of a translation library with a new number, which is used in file names to avoid collisions.
            It has to survive emulation reset, so the file names remain unique.
        */
        private static int CpuCounter = 0;

        [Export]
        private IntPtr Reallocate(IntPtr oldPointer, IntPtr newSize)
        {
            return memoryManager.Reallocate(oldPointer, newSize);
        }

        private void UpdateBlockBeginHookPresent()
        {
            TlibSetBlockBeginHookPresent((blockBeginInternalHook != null || blockBeginUserHook != null || IsSingleStepMode || hooks.IsAnyInactive) ? 1u : 0u);
        }

        private void ReactivateHooks() => hooks.Reactivate();

        [Export]
        private uint IsInDebugMode()
        {
            return InDebugMode ? 1u : 0u;
        }

        [Export]
        private void LogDisassembly(ulong pc, uint size, uint flags)
        {
            if(LogFile == null)
            {
                return;
            }
            if(Disassembler == null)
            {
                return;
            }

            if(!TryTranslateAddress(pc, MpuAccess.InstructionFetch, out var phy))
            {
                this.Log(LogLevel.Warning, "Failed to translate address 0x{0:X} while trying to log instruction disassembly", pc);
                return;
            }
            var symbol = Bus.FindSymbolAt(pc, this);
            var tab = Bus.ReadBytes(phy, (int)size, true, context: this);
            Disassembler.DisassembleBlock(pc, tab, flags, out var disas);

            if(disas == null)
            {
                return;
            }

            using(var file = File.AppendText(LogFile))
            {
                file.WriteLine("-------------------------");
                if(size > 0)
                {
                    file.Write("IN: {0} ", symbol ?? string.Empty);
                    if(phy != pc)
                    {
                        file.WriteLine("(physical: 0x{0:x8}, virtual: 0x{1:x8})", phy, pc);
                    }
                    else
                    {
                        file.WriteLine("(address: 0x{0:x8})", phy);
                    }
                }
                else
                {
                    // special case when disassembling magic addresses in Cortex-M
                    file.WriteLine("Magic PC value detected: 0x{0:x8}", flags > 0 ? pc | 1 : pc);
                }

                file.WriteLine(string.IsNullOrWhiteSpace(disas) ? string.Format("Cannot disassemble from 0x{0:x8} to 0x{1:x8}", pc, pc + size) : disas);
                file.WriteLine(string.Empty);
            }
        }

        /// <summary>
        /// See <see cref="CPUCore.MultiprocessingId" /> for explanation on how this property should be interpreted and used.
        /// Here, we can propagate this value to translation library, e.g. so it can be reflected in CPU's registers
        /// </summary>
        [Export]
        private uint GetMpIndex()
        {
            return MultiprocessingId;
        }

        private void TlibSetIrqWrapped(int number, bool state)
        {
            var decodedInterrupt = DecodeInterrupt(number);
            if(!decodedIrqs.TryGetValue(decodedInterrupt, out var irqs))
            {
                irqs = new HashSet<int>();
                decodedIrqs.Add(decodedInterrupt, irqs);
            }
            this.Log(LogLevel.Noisy, "Setting CPU IRQ #{0} to {1}", number, state);
            if(state)
            {
                irqs.Add(number);
                TlibSetIrq((int)decodedInterrupt, 1);
            }
            else
            {
                irqs.Remove(number);
                if(irqs.Count == 0)
                {
                    TlibSetIrq((int)decodedInterrupt, 0);
                }
            }
        }

        private void DeactivateHooks(ulong address) => hooks.DeactivateHooks(address);

        [Export]
        private IntPtr Allocate(IntPtr size)
        {
            return memoryManager.Allocate(size);
        }

        private CpuThreadPauseGuard ObtainPauseGuardForWriting(ulong address, SysbusAccessWidth width, ulong value)
        {
            pauseGuard.InitializeForWriting(address, width, value);
            return pauseGuard;
        }

        private void RebuildMemoryMappings()
        {
            checked
            {
                var hostBlocks = currentMappings.Where(x => x.Touched).Select(x => x.Segment)
                    .Select(x => new HostMemoryBlock { Start = x.StartingOffset, Size = x.Size, HostPointer = x.Pointer })
                    .OrderBy(x => x.HostPointer.ToInt64()).ToArray();
                if(hostBlocks.Length > 0)
                {
                    var blockBuffer = memoryManager.Allocate(new IntPtr(Marshal.SizeOf(typeof(HostMemoryBlock)) * hostBlocks.Length));
                    BlitArray(blockBuffer, hostBlocks.OrderBy(x => x.HostPointer.ToInt64()).Cast<dynamic>().ToArray());
                    RenodeSetHostBlocks(blockBuffer, hostBlocks.Length);
                    memoryManager.Free(blockBuffer);
                    this.NoisyLog("Memory mappings rebuilt, there are {0} host blocks now.", hostBlocks.Length);
                }
            }
        }

        [Export]
        private void TouchHostBlock(ulong offset)
        {
            this.NoisyLog("Trying to find the mapping for offset 0x{0:X}.", offset);
            var mapping = currentMappings.FirstOrDefault(x => x.Segment.StartingOffset <= offset && offset <= x.Segment.StartingOffset + (x.Segment.Size - 1));
            if(mapping == null)
            {
                throw new InvalidOperationException(string.Format("Could not find mapped segment for offset 0x{0:X}.", offset));
            }
            mapping.Segment.Touch();
            mapping.Touched = true;
            RebuildMemoryMappings();
        }

        private void Init(bool afterDeserialization = false)
        {
            memoryManager = new SimpleMemoryManager(this);
            isPaused = true;

            onTranslationBlockFetch = OnTranslationBlockFetch;

            // PowerPC always uses the big-endian translation library
            var endianSuffix = (Endianness == Endianess.BigEndian || Architecture.StartsWith("ppc")) ? "be" : "le";
            var libraryResource = string.Format("Antmicro.Renode.translate-{0}-{1}.so", Architecture, endianSuffix);
            foreach(var assembly in AppDomain.CurrentDomain.GetAssemblies())
            {
                if(assembly.TryFromResourceToTemporaryFile(libraryResource, out libraryFile, $"{CpuCounter}-{libraryResource}"))
                {
                    break;
                }
            }

            Interlocked.Increment(ref CpuCounter);

            if(libraryFile == null)
            {
                throw new ConstructionException($"Cannot find library {libraryResource}");
            }

            binder = new NativeBinder(this, libraryFile);
            MaximumBlockSize = DefaultMaximumBlockSize;

            // Need to call these before initializing TCG, so to save us an immediate TB flush
            // Note, that there is an additional hard limit within the translation library itself,
            // that can prevent setting the new size of translation cache, with a warning log
            var translationCacheSizeMin = (ulong)ConfigurationManager.Instance.Get("translation", "min-tb-size", DefaultMinimumTranslationCacheSize);
            var translationCacheSizeMax = (ulong)ConfigurationManager.Instance.Get("translation", "max-tb-size", DefaultMaximumTranslationCacheSize);
            TlibSetTranslationCacheConfiguration(translationCacheSizeMin, translationCacheSizeMax);

            var result = TlibInit(Model);
            if(result == -1)
            {
                throw new ConstructionException("Unknown CPU type");
            }
            if(cpuState != null)
            {
                var statePtr = TlibExportState();
                Marshal.Copy(cpuState, 0, statePtr, cpuState.Length);
                AfterLoad(statePtr);
            }
            if(externalMmuState != null)
            {
                var externalMmuStatePtr = TlibExportExternalMmuState();
                Marshal.Copy(externalMmuState, 0, externalMmuStatePtr, externalMmuState.Length);
            }

            atomicId = TlibAtomicMemoryStateInit(AtomicMemoryStatePointer, atomicId);
            if(atomicId == -1)
            {
                throw new ConstructionException("Failed to initialize atomic state, see the log for details");
            }

            var deserializationSuffix = afterDeserialization ? " after deserialization" : "";
            this.DebugLog("Initializing store table of size {0} bits{1}...", StoreTableBits, deserializationSuffix);
            TlibStoreTableInit(StoreTablePointer, (byte)StoreTableBits, afterDeserialization ? 1 : 0);
            this.DebugLog("Initialized store table of size {0} bits{1}!", StoreTableBits, deserializationSuffix);

            HandleRamSetup();
            ActivateNewHooks();
            CyclesPerInstruction = 1;
        }

        [Export]
        private void ReportAbort(string message)
        {
            this.Log(LogLevel.Error, "CPU abort [PC=0x{0:X}]: {1}.", PC.RawValue, message);
            /* If the trace writer runs asynchronyously, we need to disable it.
             * Otherwise it might catch the CpuAbortException when we cross the tlib boundary
             * since the tracer can read emulated CPU registers (tlib callbacks)
             * and catch the exception there, before the CPU thread does.
             */
            ExecutionTracerExtensions.DisableExecutionTracing(this);
            throw new CpuAbortException(message);
        }

        private void HandleRamSetup()
        {
            foreach(var mapping in currentMappings)
            {
                var range = mapping.Segment.GetRange();
                if(!disabledMemory.ContainsOverlappingRange(range))
                {
                    SetAccessMethod(range, asMemory: true);
                }
            }
        }

        [Export]
        private void OnTranslationCacheSizeChange(ulong realSize)
        {
            this.Log(LogLevel.Debug, "Translation cache size was corrected to {0}B ({1}B).", Misc.NormalizeBinary(realSize), realSize);
        }

        private void OnTranslationBlockFetch(ulong offset)
        {
            var info = Bus.FindSymbolAt(offset, this);
            if(info != string.Empty)
            {
                info = " - " + info;
            }
            this.Log(LogLevel.Info, "Fetching block @ 0x{0:X8}{1}", offset, info);
        }

        [Export]
        private IntPtr GetDirty(IntPtr size)
        {
            var dirtyAddressesList = machine.GetNewDirtyAddressesForCore(this);
            var newAddressesCount = dirtyAddressesList.Length;

            if(newAddressesCount > 0)
            {
                dirtyAddressesPtr = memoryManager.Reallocate(dirtyAddressesPtr, new IntPtr(newAddressesCount * 8));
                Marshal.Copy(dirtyAddressesList, 0, dirtyAddressesPtr, newAddressesCount);
            }
            Marshal.WriteInt64(size, newAddressesCount);

            return dirtyAddressesPtr;
        }

        [Export]
        private void OnMassBroadcastDirty(IntPtr arrayStart, int size)
        {
            var tempArray = new long[size];
            Marshal.Copy(arrayStart, tempArray, 0, size);
            machine.AppendDirtyAddresses(this, tempArray);
        }

        [Export]
        private void OnMemoryAccess(ulong pc, uint operation, ulong virtualAddress, uint width, ulong value)
        {
            // We don't care if translation fails here (the address is unchanged in this case)
            TryTranslateAddress(virtualAddress, Misc.MemoryOperationToMpuAccess((MemoryOperation)operation), out var physicalAddress);
            memoryAccessHook?.Invoke(pc, (MemoryOperation)operation, virtualAddress, physicalAddress, width, value);
        }

        private void RemoveHookAtInterruptBegin(Action<ulong> hook)
        {
            interruptBeginHook -= hook;
            if(interruptBeginHook == null)
            {
                TlibSetInterruptBeginHookPresent(0u);
            }
        }

        [Export]
        private ulong GetTotalElapsedCycles()
        {
            return this.ElapsedCycles;
        }

        [Export]
        private uint IsMemoryDisabled(ulong start, ulong size)
        {
            return disabledMemory.ContainsOverlappingRange(start.By(size)) ? 1u : 0u;
        }

        /// <remarks>
        /// This method should be called from tlib only, and never from C#, since it uses `ObtainGenericPauseGuard`
        /// see: <see cref="ObtainGenericPauseGuard" /> for more information.
        /// </remarks>
        [Export]
        private void OnWfiStateChange(int isInWfi)
        {
            using(ObtainGenericPauseGuard())
            {
                wfiStateChangeHook?.Invoke(isInWfi > 0);
            }
        }

        /// <remarks>
        /// This method should be called from tlib only, and never from C#, since it uses `ObtainGenericPauseGuard`
        /// see: <see cref="ObtainGenericPauseGuard" /> for more information.
        /// </remarks>
        [Export]
        private uint OnBlockBegin(ulong address, uint size)
        {
            ReactivateHooks();

            using(ObtainGenericPauseGuard())
            {
                blockBeginInternalHook?.Invoke(address, size);
                blockBeginUserHook?.Invoke(address, size);
            }

            return (currentHaltedState || isPaused) ? 0 : 1u;
        }

        [Export]
        private void OnInterruptEnd(ulong interruptIndex)
        {
            interruptEndHook?.Invoke(interruptIndex);
        }

        [Export]
        private int MmuFaultExternalHandler(ulong address, int accessType, ulong rawWindowId, int firstTry)
        {
            this.Log(LogLevel.Noisy, "External MMU fault at 0x{0:X} when trying to access as {1}", address, (AccessType)accessType);

            var isFirstTry = firstTry == 1;
            var windowId = rawWindowId == ulong.MaxValue ? null : (ulong?)rawWindowId;
            if(windowId == null && !isFirstTry)
            {
                this.Log(LogLevel.Error, "MMU fault - the address 0x{0:X} is not specified in any of the existing ranges", address);
            }
            var shouldRetry = mmuFaultHook?.Invoke(address, (AccessType)accessType, windowId, isFirstTry) ?? false;
            return shouldRetry ? 1 : 0;
        }

        [Export]
        private void OnInterruptBegin(ulong interruptIndex)
        {
            interruptBeginHook?.Invoke(interruptIndex);
        }

        private void BlitArray(IntPtr targetPointer, dynamic[] structures)
        {
            var count = structures.Count();
            if(count == 0)
            {
                return;
            }
            var structureSize = Marshal.SizeOf(structures.First());
            var currentPtr = targetPointer;
            for(var i = 0; i < count; i++)
            {
                Marshal.StructureToPtr(structures[i], currentPtr + i * structureSize, false);
            }
        }

        [Export]
        private void InvalidateTbInOtherCpus(IntPtr start, IntPtr end)
        {
            var otherCpus = machine.SystemBus.GetCPUs().OfType<TranslationCPU>().Where(x => x != this);
            foreach(var cpu in otherCpus)
            {
                cpu.OrderTranslationBlocksInvalidation(start, end);
            }
        }

        private void RemoveHookAtInterruptEnd(Action<ulong> hook)
        {
            interruptEndHook -= hook;
            if(interruptEndHook == null)
            {
                TlibSetInterruptEndHookPresent(0u);
            }
        }

        private void ValidateMemoryRangeAndThrow(Range range)
        {
            var pageSize = TlibGetPageSize();
            var startUnaligned = (range.StartAddress % pageSize) != 0;
            var sizeUnaligned = (range.Size % pageSize) != 0;
            if(startUnaligned || sizeUnaligned)
            {
                throw new RecoverableException($"Could not register memory at offset 0x{range.StartAddress:X} and size 0x{range.Size:X} - the {(startUnaligned ? "registration address" : "size")} has to be aligned to guest page size 0x{pageSize:X}.");
            }
        }

        private CpuThreadPauseGuard ObtainPauseGuardForReading(ulong address, SysbusAccessWidth width)
        {
            pauseGuard.InitializeForReading(address, width);
            return pauseGuard;
        }

        private void InvokeInCpuThreadSafely(Action a)
        {
            actionsToExecuteOnCpuThread.Enqueue(a);
        }

        [PreSerialization]
        private void PrepareState()
        {
            var statePtr = TlibExportState();
            BeforeSave(statePtr);
            cpuState = new byte[TlibGetStateSize()];
            Marshal.Copy(statePtr, cpuState, 0, cpuState.Length);
            var externalMmuStatePtr = TlibExportExternalMmuState();
            if(externalMmuStatePtr != IntPtr.Zero)
            {
                externalMmuState = new byte[TlibGetExternalMmuStateSize()];
                Marshal.Copy(externalMmuStatePtr, externalMmuState, 0, externalMmuState.Length);
            }
        }

        [PostSerialization]
        private void FreeState()
        {
            cpuState = null;
            externalMmuState = null;
        }

        [LatePostDeserialization]
        private void RestoreState()
        {
            Init(afterDeserialization: true);
            // TODO: state of the reset events
            FreeState();
            if(memoryAccessHook != null)
            {
                // Repeat memory hook enable to make sure that the tcg context is set not to use the tlb
                TlibOnMemoryAccessEventEnabled(1);
            }
        }

        private void LogCpuInterruptBegin(ulong exceptionIndex)
        {
            this.Log(LogLevel.Info, "Begin of the interrupt: {0}", GetExceptionDescription(exceptionIndex));
        }

        /// <remarks>
        /// This method should be called from tlib only, and never from C#, since it uses `ObtainGenericPauseGuard`
        /// see: <see cref="ObtainGenericPauseGuard" /> for more information.
        /// </remarks>
        [Export]
        private void OnBlockFinished(ulong pc, uint executedInstructions)
        {
            using(ObtainGenericPauseGuard())
            {
                blockFinishedHook?.Invoke(pc, executedInstructions);
            }
        }

        private void SetInternalHookAtBlockBegin(Action<ulong, uint> hook)
        {
            using(machine?.ObtainPausedState(true))
            {
                if((hook == null) ^ (blockBeginInternalHook == null))
                {
                    ClearTranslationCache();
                }
                blockBeginInternalHook = hook;
                UpdateBlockBeginHookPresent();
            }
        }

        private bool AssertMmuEnabled()
        {
            if(externalMmuPosition == ExternalMmuPosition.None)
            {
                throw new RecoverableException("External MMU not enabled");
            }
            return true;
        }

        /* Currently, due to the used types, 64 bit targets will always pass this check.
        Also on such platforms unary overflow is not possible */
        private bool AssertMmuWindowAddressInRange(ulong address, bool inclusiveRange = false)
        {
            ulong maxValue;
            switch(this.bitness)
            {
            case CpuBitness.Bits32:
                maxValue = UInt32.MaxValue;
                break;
            case CpuBitness.Bits64:
                maxValue = UInt64.MaxValue;
                break;
            default:
                throw new ArgumentException("Unexpected value of the CpuBitness");
            }

            if(inclusiveRange && address != 0)
            {
                address -= 1;
            }

            if(address > maxValue)
            {
                throw new RecoverableException($"Address is outside of the possible range. Maximum value: {maxValue}");
            }

            return true;
        }

        private void SetAccessMethod(Range range, bool asMemory)
        {
            using(machine?.ObtainPausedState(true))
            {
                ValidateMemoryRangeAndThrow(range);
                if(!mappedMemory.ContainsWholeRange(range))
                {
                    throw new RecoverableException(
                        $"Tried to set mapped memory access method at {range} which isn't mapped memory in CPU: {this.GetName()}"
                    );
                }

                if(asMemory)
                {
                    TlibMapRange(range.StartAddress, range.Size);
                }
                else
                {
                    TlibUnmapRange(range.StartAddress, range.EndAddress);
                }
            }
        }

        private void LogCpuInterruptEnd(ulong exceptionIndex)
        {
            this.Log(LogLevel.Info, "End of the interrupt: {0}", GetExceptionDescription(exceptionIndex));
        }

        private IntPtr AtomicMemoryStatePointer =>
            useMachineAtomicState
                ? machine.AtomicMemoryStatePointer
                : localAtomicState.AtomicMemoryStatePointer;

        private IntPtr StoreTablePointer =>
            useMachineAtomicState
                ? machine.StoreTablePointer
                : localAtomicState.StoreTablePointer;

        private int StoreTableBits =>
            useMachineAtomicState
                ? machine.StoreTableBits
                : localAtomicState.StoreTableBits;

        private ExternalMmuPosition externalMmuPosition;

        /// <summary>
        /// <see cref="atomicId" /> acts as a binder between the CPU and atomic state.
        /// It's used to restore the atomic state after deserialization
        /// </summary>
        private int atomicId;
        private byte[] cpuState;
        private byte[] externalMmuState;

        [Transient]
        private TranslationBlockFetchCallback onTranslationBlockFetch;

        [Transient]
        private IntPtr dirtyAddressesPtr = IntPtr.Zero;
        private TlibExecutionResult lastTlibResult;
        private int? slot;

        private bool logTranslationBlockFetchEnabled;

        [Transient]
        private List<IntPtr> addressesToInvalidate;

        private Action<ulong, uint> blockBeginInternalHook;

        [Transient]
        private string libraryFile;
        private Action<ulong, uint> blockBeginUserHook;
        private Action<ulong> interruptBeginHook;
        private Action<ulong> interruptEndHook;
        private ExternalMmuFaultHook mmuFaultHook;
        private MemoryAccessHook memoryAccessHook;
        private Action<bool> wfiStateChangeHook;

        private List<SegmentMapping> currentMappings;

        [Transient]
        private NativeBinder binder;

        private bool logTranslatedBlocks;
        private bool isInterruptLoggingEnabled;
        private LogFunctionNamesState? logFunctionNamesCurrentState;
        private Action<ulong, uint> blockFinishedHook;

        [Transient]
        private SimpleMemoryManager memoryManager;

        [Transient]
        private LLVMAssembler assembler;

        [Transient]
        private LLVMDisassembler disassembler;

        [Transient]
        private bool disposed;

        private string logFile;
        private readonly HookDescriptorBase hooks;

        private readonly AtomicState localAtomicState;
        private readonly bool useMachineAtomicState;

        private readonly Dictionary<Interrupt, HashSet<int>> decodedIrqs;

        // 649:  Field '...' is never assigned to, and will always have its default value null
#pragma warning disable 649
        [Import]
        private readonly Action<uint> TlibSetBlockBeginHookPresent;

        [Import]
        private readonly Action<ulong, ulong> TlibSetMmuWindowStart;

        [Import]
        private readonly Action<ulong> TlibResetMmuWindow;

        [Import]
        private readonly Action<ulong> TlibResetMmuWindowsCoveringAddress;

        [Import]
        private readonly Action TlibResetAllMmuWindows;

        [Import]
        private readonly Func<uint, ulong> TlibAcquireMmuWindow;

        [Import]
        private readonly Action<uint> TlibEnableExternalWindowMmu;

        [Import]
        private readonly Action<uint> TlibRaiseException;

        [Import]
        private readonly Func<uint> TlibGetMmuWindowsCount;

        [Import(UseExceptionWrapper = false)]
        private readonly Action TlibUnwind;

        [Import]
        private readonly Action<ulong> TlibClearPageIoAccessed;

        [Import]
        private readonly Action<ulong> TlibSetPageIoAccessed;

        [Import]
        private readonly Action TlibCleanWfiProcState;

        [Import]
        private readonly Action<int> TlibOnMemoryAccessEventEnabled;

        [Import]
        private readonly Func<ulong> TlibGetTotalExecutedInstructions;

        [Import]
        private readonly Action<uint> TlibSetInterruptEndHookPresent;

        [Import]
        private readonly Action<uint> TlibSetCpuWfiStateChangeHookPresent;

        [Import]
        private readonly Action<uint> TlibSetInterruptBeginHookPresent;

        [Import]
        private readonly Func<uint> TlibGetCurrentTbDisasFlags;

        [Import]
        private readonly Action<ulong, ulong, uint> TlibSetMmuWindowEnd;

        [Import]
        private readonly Action<ulong, ulong> TlibSetMmuWindowAddend;

        [Import]
        private readonly Action<ulong, uint> TlibSetWindowPrivileges;

        [Import]
        private readonly Action<IntPtr> TlibAfterLoad;

        [Import]
        private readonly Action TlibOnLeavingResetState;

        [Import(UseExceptionWrapper = false)] // Not wrapped for performance
        private readonly Func<ulong> TlibGetCpuStateForMemoryTransaction;

        [Import]
        private readonly Action<int> TlibSetBroadcastDirty;

        [Import]
        private readonly Func<ulong, uint> TlibGetWindowPrivileges;

        [Import]
        private readonly Func<ulong, ulong> TlibGetMmuWindowEnd;

        [Import]
        private readonly Func<ulong, ulong> TlibGetMmuWindowStart;

        [Import]
        private readonly Action<IntPtr> TlibBeforeSave;

        [Import]
        private readonly Func<ulong, ulong> TlibGetMmuWindowAddend;

        [Import]
        private readonly Action<uint> TlibSetMillicyclesPerInstruction;

        [Import]
        private readonly Func<int> TlibGetStateSize;

        [Import]
        private readonly Func<int> TlibGetExternalMmuStateSize;

        [Import]
        private readonly Action TlibReset;

        [Import]
        private readonly Action TlibDispose;

        [Import]
        private readonly Func<string, int> TlibInit;

        [Import]
        private readonly Func<uint> TlibGetSyncPcEveryInstructionDisabled;

        [Import]
        private readonly Action<uint> TlibSetBlockFinishedHookPresent;

        [Import]
        private readonly Func<uint> TlibGetTbCacheEnabled;

        [Import]
        private readonly Action<uint> TlibSetTbCacheEnabled;

        [Import]
        private readonly Func<uint> TlibGetChainingEnabled;

        [Import]
        private readonly Action<uint> TlibSetChainingEnabled;

        [Import]
        private readonly Func<int, int> TlibExecute;

        [Import]
        private readonly Func<IntPtr, int, int> TlibAtomicMemoryStateInit;

        [Import]
        private readonly Func<IntPtr, byte, int, int> TlibStoreTableInit;

        [Import]
        private readonly Action<uint> TlibSetSyncPcEveryInstructionDisabled;

        [Import]
        private readonly Action<ulong, ulong> TlibMapRange;

        [Import]
        private readonly Action<int> TlibSetOnBlockTranslationEnabled;

        [Import]
        private readonly Func<uint, uint> TlibSetMaximumBlockSize;

        [Import]
        private readonly Action<ulong> TlibFlushPage;

        [Import]
        private readonly Action<bool> TlibFlushTlb;

        [Import]
        private readonly Func<uint> TlibGetPageSize;

        [Import]
        private readonly Func<uint> TlibGetMaximumBlockSize;

        [Import]
        private readonly Func<uint> TlibGetMillicyclesPerInstruction;

        [Import]
        private readonly Func<int> TlibRestoreContext;

        [Import]
        private readonly Func<IntPtr> TlibExportState;

        [Import]
        private readonly Func<IntPtr> TlibExportExternalMmuState;

        [Import]
        private readonly Action TlibInvalidateTranslationCache;

        [Import]
        private readonly Action<IntPtr> RenodeAttachLogTranslationBlockFetch;

        [Import]
        private readonly Action<ulong> TlibAddBreakpoint;

        [Import]
        private readonly Func<uint> TlibIsIrqSet;

        [Import]
        private readonly Action<int, int> TlibSetIrq;

        [Import]
        private readonly Action RenodeFreeHostBlocks;

        [Import]
        private readonly Action<IntPtr, int> RenodeSetHostBlocks;

        [Import]
        private readonly Action<IntPtr, ulong> TlibInvalidateTranslationBlocks;

        [Import]
        private readonly Func<ulong, ulong, uint> TlibIsRangeMapped;

        [Import]
        private readonly Action<ulong, ulong, uint> TlibRegisterAccessFlagsForRange;

        [Import]
        private readonly Action<ulong, ulong> TlibUnmapRange;

        [Import]
        private readonly Action<ulong> TlibRemoveBreakpoint;

        [Import]
        private readonly Action<ulong, ulong> TlibSetTranslationCacheConfiguration;

        [Import]
        private readonly Action<ulong, ulong, ulong> TlibEnableReadCache;
#pragma warning restore 649

        private readonly ConcurrentQueue<Action> actionsToExecuteOnCpuThread = new ConcurrentQueue<Action>();

        // TODO
        private readonly object lck = new object();
        private readonly MinimalRangesCollection mappedMemory = new MinimalRangesCollection();
        private readonly CpuThreadPauseGuard pauseGuard;

        private readonly MinimalRangesCollection disabledMemory = new MinimalRangesCollection();
        private const int DefaultMaximumTranslationCacheSize = 512 * 1024 * 1024; // 512 MiB
        private const int DefaultMinimumTranslationCacheSize = 32 * 1024 * 1024; // 32 MiB

        private const int DefaultMaximumBlockSize = 0x7FF;

        protected sealed class CpuThreadPauseGuard : IDisposable
        {
            public CpuThreadPauseGuard(TranslationCPU parent)
            {
                guard = new ThreadLocal<object>();
                this.parent = parent;
            }

            public void OrderPause()
            {
                if(active && guard.Value == null)
                {
                    throw new InvalidOperationException("Trying to order pause without prior guard initialization on this thread.");
                }
            }

            public bool RequestTranslationBlockRestart(bool quiet = false)
            {
                if(guard.Value == null)
                {
                    if(!quiet)
                    {
                        parent.Log(LogLevel.Error, "Trying to request translation block restart without prior guard initialization on this thread.");
                    }
                    return false;
                }
                restartTranslationBlock = true;
                return true;
            }

            public void Enter()
            {
                active = true;
            }

            public void Leave()
            {
                active = false;
            }

            public void Initialize()
            {
                guard.Value = new object();
            }

            public void InitializeForWriting(ulong address, SysbusAccessWidth width, ulong value)
            {
                InterruptTransaction = !ExecuteWatchpoints(address, width, value);
            }

            public void InitializeForReading(ulong address, SysbusAccessWidth width)
            {
                InterruptTransaction = !ExecuteWatchpoints(address, width, null);
            }

            public bool InterruptTransaction { get; private set; }

            private bool ExecuteWatchpoints(ulong address, SysbusAccessWidth width, ulong? value)
            {
                Initialize();
                if(!parent.machine.SystemBus.TryGetWatchpointsAt(address, value.HasValue ? Access.Write : Access.Read, out var watchpoints))
                {
                    return true;
                }

                /*
                    * In general precise pause works as follows:
                    * - translation libraries execute an instruction that reads/writes to/from memory
                    * - the execution is then transferred to the system bus (to process memory access)
                    * - we check whether there are any hooks registered for the accessed address (TryGetWatchpointsAt)
                    * - if there are (and we hit them for the first time) we call them and then invalidate the block and issue retranslation of the code at current PC
                    * - we exit the cpu loop so that newly translated block will be executed now
                    * - the next time we hit them we do nothing
                */

                var anyEnabled = false;
                var alreadyUpdated = false;
                foreach(var enabledWatchpoint in watchpoints.Where(x => x.Enabled))
                {
                    enabledWatchpoint.Enabled = false;
                    if(!alreadyUpdated && parent.UpdateContextOnLoadAndStore)
                    {
                        parent.TlibRestoreContext();
                        alreadyUpdated = true;
                    }

                    // for reading value is always set to 0
                    enabledWatchpoint.Invoke(parent, address, width, value ?? 0);
                    anyEnabled = true;
                }

                if(anyEnabled)
                {
                    parent.TlibRequestTranslationBlockInterrupt(1);

                    // tell sysbus to cancel the current transaction and return immediately
                    return false;
                }
                else
                {
                    foreach(var disabledWatchpoint in watchpoints)
                    {
                        disabledWatchpoint.Enabled = true;
                    }
                }

                return true;
            }

            void IDisposable.Dispose()
            {
                if(restartTranslationBlock)
                {
                    restartTranslationBlock = false;
                    if(parent.UpdateContextOnLoadAndStore)
                    {
                        parent.TlibRestoreContext();
                    }
                    parent.TlibRequestTranslationBlockInterrupt(0);
                    return;
                }
                guard.Value = null;
            }

            private bool active;
            private bool restartTranslationBlock;

            [Constructor]
            private readonly ThreadLocal<object> guard;

            private readonly TranslationCPU parent;
        }

        protected enum TlibExecutionResult : ulong
        {
            Ok = 0x10000,
            WaitingForInterrupt = 0x10001,
            StoppedAtBreakpoint = 0x10002,
            StoppedAtWatchpoint = 0x10004,
            ReturnRequested = 0x10005,
            ExternalMmuFault = 0x10006,
        }

        protected enum Interrupt
        {
            Hard            = 1 << 1,
            TargetExternal0 = 1 << 3,
            TargetExternal1 = 1 << 4,
            TargetExternal2 = 1 << 6,
            TargetExternal3 = 1 << 9,
        }

        private class HookDescriptor : HookDescriptorBase
        {
            public HookDescriptor(ICpuSupportingGdb cpu) : base(cpu)
            {
            }

            protected override void HookStateChangedCallback() => (cpu as TranslationCPU).UpdateBlockBeginHookPresent();

            protected override void AddBreakpoint(ulong address) => (cpu as TranslationCPU).TlibAddBreakpoint(address);

            protected override void RemoveBreakpoint(ulong address) => (cpu as TranslationCPU).TlibRemoveBreakpoint(address);
        }

        private class SimpleMemoryManager
        {
            public SimpleMemoryManager(TranslationCPU parent)
            {
                this.parent = parent;
                ourPointers = new ConcurrentDictionary<IntPtr, long>();
            }

            public void CheckIfAllIsFreed()
            {
                if(!ourPointers.IsEmpty)
                {
                    parent.Log(LogLevel.Warning, "Some memory allocated by the translation library was not freed - {0}B left allocated. This might indicate a memory leak. Cleaning up...", Misc.NormalizeBinary(allocated));
                    foreach(var ptr in ourPointers.Keys)
                    {
                        Marshal.FreeHGlobal(ptr);
                    }
                }
            }

            public IntPtr Allocate(IntPtr size)
            {
                var ptr = Marshal.AllocHGlobal(size);
                var sizeNormalized = Misc.NormalizeBinary((double)size);
                if(!ourPointers.TryAdd(ptr, (long)size))
                {
                    throw new InvalidOperationException($"Trying to allocate a {sizeNormalized}B pointer that already exists is the memory database.");
                }
                Interlocked.Add(ref allocated, (long)size);
                parent.NoisyLog("Allocated {0}B pointer at 0x{1:X}.", sizeNormalized, ptr);
                PrintAllocated();
                return ptr;
            }

            public IntPtr Reallocate(IntPtr oldPointer, IntPtr newSize)
            {
                if(oldPointer == IntPtr.Zero)
                {
                    return Allocate(newSize);
                }
                if(newSize == IntPtr.Zero)
                {
                    Free(oldPointer);
                    return IntPtr.Zero;
                }
                if(!ourPointers.TryRemove(oldPointer, out var oldSize))
                {
                    throw new InvalidOperationException($"Trying to reallocate a pointer at 0x{oldPointer:X} which wasn't allocated by this memory manager.");
                }
                var ptr = Marshal.ReAllocHGlobal(oldPointer, newSize);
                parent.NoisyLog("Reallocated a pointer: old size {0}B at 0x{1:X}, new size {2}B at 0x{3:X}.", Misc.NormalizeBinary(oldSize), oldPointer, Misc.NormalizeBinary((double)newSize), ptr);
                Interlocked.Add(ref allocated, (long)newSize - oldSize);
                ourPointers.TryAdd(ptr, (long)newSize);
                return ptr;
            }

            public void Free(IntPtr ptr)
            {
                if(!ourPointers.TryRemove(ptr, out var oldSize))
                {
                    throw new InvalidOperationException($"Trying to free a pointer at 0x{ptr:X} which wasn't allocated by this memory manager.");
                }
                parent.NoisyLog("Deallocated a {0}B pointer at 0x{1:X}.", Misc.NormalizeBinary(oldSize), ptr);
                Marshal.FreeHGlobal(ptr);
                Interlocked.Add(ref allocated, -oldSize);
            }

            public long Allocated
            {
                get
                {
                    return allocated;
                }
            }

            private void PrintAllocated()
            {
                parent.NoisyLog("Allocated is now {0}B.", Misc.NormalizeBinary(Interlocked.Read(ref allocated)));
            }

            private long allocated;

            private readonly ConcurrentDictionary<IntPtr, long> ourPointers;
            private readonly TranslationCPU parent;
        }

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        private struct HostMemoryBlock
        {
            public ulong Start;
            public ulong Size;
            public IntPtr HostPointer;
        }

        private struct TranslationCPUState
        {
            public TranslationCPUState(LogFunctionNamesState? logFunctionNamesState, bool isInterruptLoggingEnabled)
            {
                LogFunctionNamesState = logFunctionNamesState;
                IsInterruptLoggingEnabled = isInterruptLoggingEnabled;
            }

            public LogFunctionNamesState? LogFunctionNamesState { get; private set; }

            public bool IsInterruptLoggingEnabled { get; private set; }
        }

        private struct LogFunctionNamesState
        {
            public LogFunctionNamesState(string spaceSeparatedPrefixes, bool removeDuplicates, bool useFunctionSymbolsOnly)
            {
                this.SpaceSeparatedPrefixes = spaceSeparatedPrefixes;
                this.RemoveDuplicates = removeDuplicates;
                this.UseFunctionSymbolsOnly = useFunctionSymbolsOnly;
            }

            public string SpaceSeparatedPrefixes { get; private set; }

            public bool RemoveDuplicates { get; private set; }

            public bool UseFunctionSymbolsOnly { get; private set; }
        }

        private delegate void TranslationBlockFetchCallback(ulong pc);

        protected static readonly Exception InvalidInterruptNumberException = new InvalidOperationException("Invalid interrupt number.");
    }
}
